"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupService = void 0;
const common_1 = require("@nestjs/common");
const luxon_1 = require("luxon");
const node_path_1 = __importDefault(require("node:path"));
const semver_1 = __importDefault(require("semver"));
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const misc_1 = require("../utils/misc");
let BackupService = class BackupService extends base_service_1.BaseService {
    backupLock = false;
    async onConfigInit({ newConfig: { backup: { database }, }, }) {
        this.backupLock = await this.databaseRepository.tryLock(enum_1.DatabaseLock.BackupDatabase);
        if (this.backupLock) {
            this.cronRepository.create({
                name: 'backupDatabase',
                expression: database.cronExpression,
                onTick: () => (0, misc_1.handlePromiseError)(this.jobRepository.queue({ name: enum_1.JobName.DatabaseBackup }), this.logger),
                start: database.enabled,
            });
        }
    }
    onConfigUpdate({ newConfig: { backup } }) {
        if (!this.backupLock) {
            return;
        }
        this.cronRepository.update({
            name: 'backupDatabase',
            expression: backup.database.cronExpression,
            start: backup.database.enabled,
        });
    }
    async cleanupDatabaseBackups() {
        this.logger.debug(`Database Backup Cleanup Started`);
        const { backup: { database: config }, } = await this.getConfig({ withCache: false });
        const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
        const files = await this.storageRepository.readdir(backupsFolder);
        const failedBackups = files.filter((file) => file.match(/immich-db-backup-.*\.sql\.gz\.tmp$/));
        const backups = files
            .filter((file) => {
            const oldBackupStyle = file.match(/immich-db-backup-\d+\.sql\.gz$/);
            const newBackupStyle = file.match(/immich-db-backup-\d{8}T\d{6}-v.*-pg.*\.sql\.gz$/);
            return oldBackupStyle || newBackupStyle;
        })
            .sort()
            .toReversed();
        const toDelete = backups.slice(config.keepLastAmount);
        toDelete.push(...failedBackups);
        for (const file of toDelete) {
            await this.storageRepository.unlink(node_path_1.default.join(backupsFolder, file));
        }
        this.logger.debug(`Database Backup Cleanup Finished, deleted ${toDelete.length} backups`);
    }
    async handleBackupDatabase() {
        this.logger.debug(`Database Backup Started`);
        const { database } = this.configRepository.getEnv();
        const config = database.config;
        const isUrlConnection = config.connectionType === 'url';
        const databaseParams = isUrlConnection
            ? ['--dbname', config.url]
            : [
                '--username',
                config.username,
                '--host',
                config.host,
                '--port',
                `${config.port}`,
                '--database',
                config.database,
            ];
        databaseParams.push('--clean', '--if-exists');
        const databaseVersion = await this.databaseRepository.getPostgresVersion();
        const backupFilePath = node_path_1.default.join(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), `immich-db-backup-${luxon_1.DateTime.now().toFormat("yyyyLLdd'T'HHmmss")}-v${constants_1.serverVersion.toString()}-pg${databaseVersion.split(' ')[0]}.sql.gz.tmp`);
        const databaseSemver = semver_1.default.coerce(databaseVersion);
        const databaseMajorVersion = databaseSemver?.major;
        if (!databaseMajorVersion || !databaseSemver || !semver_1.default.satisfies(databaseSemver, '>=14.0.0 <19.0.0')) {
            this.logger.error(`Database Backup Failure: Unsupported PostgreSQL version: ${databaseVersion}`);
            return enum_1.JobStatus.Failed;
        }
        this.logger.log(`Database Backup Starting. Database Version: ${databaseMajorVersion}`);
        try {
            await new Promise((resolve, reject) => {
                const pgdump = this.processRepository.spawn(`/usr/bin/pg_dumpall`, databaseParams, {
                    env: {
                        PATH: process.env.PATH,
                        PGPASSWORD: isUrlConnection ? new URL(config.url).password : config.password,
                    },
                });
                const gzip = this.processRepository.spawn(`gzip`, ['--rsyncable']);
                pgdump.stdout.pipe(gzip.stdin);
                const fileStream = this.storageRepository.createWriteStream(backupFilePath);
                gzip.stdout.pipe(fileStream);
                pgdump.on('error', (err) => {
                    this.logger.error(`Backup failed with error: ${err}`);
                    reject(err);
                });
                gzip.on('error', (err) => {
                    this.logger.error(`Gzip failed with error: ${err}`);
                    reject(err);
                });
                let pgdumpLogs = '';
                let gzipLogs = '';
                pgdump.stderr.on('data', (data) => (pgdumpLogs += data));
                gzip.stderr.on('data', (data) => (gzipLogs += data));
                pgdump.on('exit', (code) => {
                    if (code !== 0) {
                        this.logger.error(`Backup failed with code ${code}`);
                        reject(`Backup failed with code ${code}`);
                        this.logger.error(pgdumpLogs);
                        return;
                    }
                    if (pgdumpLogs) {
                        this.logger.debug(`pgdump_all logs\n${pgdumpLogs}`);
                    }
                });
                gzip.on('exit', (code) => {
                    if (code !== 0) {
                        this.logger.error(`Gzip failed with code ${code}`);
                        reject(`Gzip failed with code ${code}`);
                        this.logger.error(gzipLogs);
                        return;
                    }
                    if (pgdump.exitCode !== 0) {
                        this.logger.error(`Gzip exited with code 0 but pgdump exited with ${pgdump.exitCode}`);
                        return;
                    }
                    resolve();
                });
            });
            await this.storageRepository.rename(backupFilePath, backupFilePath.replace('.tmp', ''));
        }
        catch (error) {
            this.logger.error(`Database Backup Failure: ${error}`);
            await this.storageRepository
                .unlink(backupFilePath)
                .catch((error) => this.logger.error(`Failed to delete failed backup file: ${error}`));
            throw error;
        }
        this.logger.log(`Database Backup Success`);
        await this.cleanupDatabaseBackups();
        return enum_1.JobStatus.Success;
    }
};
exports.BackupService = BackupService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigInit', workers: [enum_1.ImmichWorker.Microservices] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], BackupService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigUpdate', server: true }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], BackupService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.DatabaseBackup, queue: enum_1.QueueName.BackupDatabase }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], BackupService.prototype, "handleBackupDatabase", null);
exports.BackupService = BackupService = __decorate([
    (0, common_1.Injectable)()
], BackupService);
//# sourceMappingURL=backup.service.js.map