"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetadataService = void 0;
exports.firstDateTime = firstDateTime;
const common_1 = require("@nestjs/common");
const exiftool_vendored_1 = require("exiftool-vendored");
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const promises_1 = require("node:fs/promises");
const node_path_1 = require("node:path");
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const database_1 = require("../utils/database");
const misc_1 = require("../utils/misc");
const tag_1 = require("../utils/tag");
const EXIF_DATE_TAGS = [
    'SubSecDateTimeOriginal',
    'SubSecCreateDate',
    'SubSecMediaCreateDate',
    'DateTimeOriginal',
    'CreationDate',
    'CreateDate',
    'MediaCreateDate',
    'DateTimeCreated',
    'GPSDateTime',
    'DateTimeUTC',
    'SonyDateTime2',
    'SourceImageCreateTime',
];
function firstDateTime(tags) {
    for (const tag of EXIF_DATE_TAGS) {
        const tagValue = tags?.[tag];
        if (tagValue instanceof exiftool_vendored_1.ExifDateTime) {
            return {
                tag,
                dateTime: tagValue,
            };
        }
        if (typeof tagValue !== 'string') {
            continue;
        }
        const exifDateTime = exiftool_vendored_1.ExifDateTime.fromEXIF(tagValue);
        if (exifDateTime) {
            return {
                tag,
                dateTime: exifDateTime,
            };
        }
    }
}
const validate = (value) => {
    if (Array.isArray(value)) {
        value = value[0];
    }
    if (typeof value === 'string') {
        return null;
    }
    if (typeof value === 'number' && (Number.isNaN(value) || !Number.isFinite(value))) {
        return null;
    }
    return value ?? null;
};
const validateRange = (value, min, max) => {
    const val = validate(value);
    if (val == null || val < min || val > max) {
        return null;
    }
    return Math.round(val);
};
const getLensModel = (exifTags) => {
    const lensModel = String(exifTags.LensID ?? exifTags.LensType ?? exifTags.LensSpec ?? exifTags.LensModel ?? '').trim();
    if (lensModel === '----') {
        return null;
    }
    if (lensModel.startsWith('Unknown')) {
        return null;
    }
    return lensModel || null;
};
let MetadataService = class MetadataService extends base_service_1.BaseService {
    async onBootstrap() {
        this.logger.log('Bootstrapping metadata service');
        await this.init();
    }
    async onShutdown() {
        await this.metadataRepository.teardown();
    }
    onConfigInit({ newConfig }) {
        this.metadataRepository.setMaxConcurrency(newConfig.job.metadataExtraction.concurrency);
    }
    onConfigUpdate({ newConfig }) {
        this.metadataRepository.setMaxConcurrency(newConfig.job.metadataExtraction.concurrency);
    }
    async init() {
        this.logger.log('Initializing metadata service');
        try {
            await this.jobRepository.pause(enum_1.QueueName.MetadataExtraction);
            await this.databaseRepository.withLock(enum_1.DatabaseLock.GeodataImport, () => this.mapRepository.init());
            await this.jobRepository.resume(enum_1.QueueName.MetadataExtraction);
            this.logger.log(`Initialized local reverse geocoder`);
        }
        catch (error) {
            this.logger.error(`Unable to initialize reverse geocoding: ${error}`, error?.stack);
            throw new Error(`Metadata service init failed`);
        }
    }
    async linkLivePhotos(asset, exifInfo) {
        if (!exifInfo.livePhotoCID) {
            return;
        }
        const otherType = asset.type === enum_1.AssetType.Video ? enum_1.AssetType.Image : enum_1.AssetType.Video;
        const match = await this.assetRepository.findLivePhotoMatch({
            livePhotoCID: exifInfo.livePhotoCID,
            ownerId: asset.ownerId,
            libraryId: asset.libraryId,
            otherAssetId: asset.id,
            type: otherType,
        });
        if (!match) {
            return;
        }
        const [photoAsset, motionAsset] = asset.type === enum_1.AssetType.Image ? [asset, match] : [match, asset];
        await Promise.all([
            this.assetRepository.update({ id: photoAsset.id, livePhotoVideoId: motionAsset.id }),
            this.assetRepository.update({ id: motionAsset.id, visibility: enum_1.AssetVisibility.Hidden }),
            this.albumRepository.removeAssetsFromAll([motionAsset.id]),
        ]);
        await this.eventRepository.emit('AssetHide', { assetId: motionAsset.id, userId: motionAsset.ownerId });
    }
    async handleQueueMetadataExtraction(job) {
        const { force } = job;
        let queue = [];
        for await (const asset of this.assetJobRepository.streamForMetadataExtraction(force)) {
            queue.push({ name: enum_1.JobName.AssetExtractMetadata, data: { id: asset.id } });
            if (queue.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await this.jobRepository.queueAll(queue);
                queue = [];
            }
        }
        await this.jobRepository.queueAll(queue);
        return enum_1.JobStatus.Success;
    }
    async handleMetadataExtraction(data) {
        const [{ metadata, reverseGeocoding }, asset] = await Promise.all([
            this.getConfig({ withCache: true }),
            this.assetJobRepository.getForMetadataExtraction(data.id),
        ]);
        if (!asset) {
            return;
        }
        const [exifTags, stats] = await Promise.all([
            this.getExifTags(asset),
            this.storageRepository.stat(asset.originalPath),
        ]);
        this.logger.verbose('Exif Tags', exifTags);
        const dates = this.getDates(asset, exifTags, stats);
        const { width, height } = this.getImageDimensions(exifTags);
        let geo = { country: null, state: null, city: null }, latitude = null, longitude = null;
        if (this.hasGeo(exifTags)) {
            latitude = Number(exifTags.GPSLatitude);
            longitude = Number(exifTags.GPSLongitude);
            if (reverseGeocoding.enabled) {
                geo = await this.mapRepository.reverseGeocode({ latitude, longitude });
            }
        }
        const exifData = {
            assetId: asset.id,
            dateTimeOriginal: dates.dateTimeOriginal,
            modifyDate: stats.mtime,
            timeZone: dates.timeZone,
            latitude,
            longitude,
            country: geo.country,
            state: geo.state,
            city: geo.city,
            fileSizeInByte: stats.size,
            exifImageHeight: validate(height),
            exifImageWidth: validate(width),
            orientation: validate(exifTags.Orientation)?.toString() ?? null,
            projectionType: exifTags.ProjectionType ? String(exifTags.ProjectionType).toUpperCase() : null,
            bitsPerSample: this.getBitsPerSample(exifTags),
            colorspace: exifTags.ColorSpace ?? null,
            make: exifTags.Make ?? exifTags?.Device?.Manufacturer ?? exifTags.AndroidMake ?? null,
            model: exifTags.Model ?? exifTags?.Device?.ModelName ?? exifTags.AndroidModel ?? null,
            fps: validate(Number.parseFloat(exifTags.VideoFrameRate)),
            iso: validate(exifTags.ISO),
            exposureTime: exifTags.ExposureTime ?? null,
            lensModel: getLensModel(exifTags),
            fNumber: validate(exifTags.FNumber),
            focalLength: validate(exifTags.FocalLength),
            description: String(exifTags.ImageDescription || exifTags.Description || '').trim(),
            profileDescription: exifTags.ProfileDescription || null,
            rating: validateRange(exifTags.Rating, -1, 5),
            livePhotoCID: (exifTags.ContentIdentifier || exifTags.MediaGroupUUID) ?? null,
            autoStackId: this.getAutoStackId(exifTags),
        };
        const promises = [
            this.assetRepository.upsertExif(exifData),
            this.assetRepository.update({
                id: asset.id,
                duration: exifTags.Duration?.toString() ?? null,
                localDateTime: dates.localDateTime,
                fileCreatedAt: dates.dateTimeOriginal ?? undefined,
                fileModifiedAt: stats.mtime,
            }),
            this.applyTagList(asset, exifTags),
        ];
        if (this.isMotionPhoto(asset, exifTags)) {
            promises.push(this.applyMotionPhotos(asset, exifTags, dates, stats));
        }
        if ((0, misc_1.isFaceImportEnabled)(metadata) && this.hasTaggedFaces(exifTags)) {
            promises.push(this.applyTaggedFaces(asset, exifTags));
        }
        await Promise.all(promises);
        if (exifData.livePhotoCID) {
            await this.linkLivePhotos(asset, exifData);
        }
        await this.assetRepository.upsertJobStatus({ assetId: asset.id, metadataExtractedAt: new Date() });
        await this.eventRepository.emit('AssetMetadataExtracted', {
            assetId: asset.id,
            userId: asset.ownerId,
            source: data.source,
        });
    }
    async handleQueueSidecar({ force }) {
        let jobs = [];
        const queueAll = async () => {
            await this.jobRepository.queueAll(jobs);
            jobs = [];
        };
        const assets = this.assetJobRepository.streamForSidecar(force);
        for await (const asset of assets) {
            jobs.push({ name: enum_1.JobName.SidecarCheck, data: { id: asset.id } });
            if (jobs.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await queueAll();
            }
        }
        await queueAll();
        return enum_1.JobStatus.Success;
    }
    async handleSidecarCheck({ id }) {
        const asset = await this.assetJobRepository.getForSidecarCheckJob(id);
        if (!asset) {
            return;
        }
        let sidecarPath = null;
        for (const candidate of this.getSidecarCandidates(asset)) {
            const exists = await this.storageRepository.checkFileExists(candidate, promises_1.constants.R_OK);
            if (!exists) {
                continue;
            }
            sidecarPath = candidate;
            break;
        }
        const isChanged = sidecarPath !== asset.sidecarPath;
        this.logger.debug(`Sidecar check found old=${asset.sidecarPath}, new=${sidecarPath} will ${isChanged ? 'update' : 'do nothing for'}  asset ${asset.id}: ${asset.originalPath}`);
        if (!isChanged) {
            return enum_1.JobStatus.Skipped;
        }
        await this.assetRepository.update({ id: asset.id, sidecarPath });
        return enum_1.JobStatus.Success;
    }
    async handleTagAsset({ assetId }) {
        await this.jobRepository.queue({ name: enum_1.JobName.SidecarWrite, data: { id: assetId, tags: true } });
    }
    async handleUntagAsset({ assetId }) {
        await this.jobRepository.queue({ name: enum_1.JobName.SidecarWrite, data: { id: assetId, tags: true } });
    }
    async handleSidecarWrite(job) {
        const { id, description, dateTimeOriginal, latitude, longitude, rating, tags } = job;
        const asset = await this.assetJobRepository.getForSidecarWriteJob(id);
        if (!asset) {
            return enum_1.JobStatus.Failed;
        }
        const tagsList = (asset.tags || []).map((tag) => tag.value);
        const sidecarPath = asset.sidecarPath || `${asset.originalPath}.xmp`;
        const exif = lodash_1.default.omitBy({
            Description: description,
            ImageDescription: description,
            DateTimeOriginal: dateTimeOriginal,
            GPSLatitude: latitude,
            GPSLongitude: longitude,
            Rating: rating,
            TagsList: tags ? tagsList : undefined,
        }, lodash_1.default.isUndefined);
        if (Object.keys(exif).length === 0) {
            return enum_1.JobStatus.Skipped;
        }
        await this.metadataRepository.writeTags(sidecarPath, exif);
        if (!asset.sidecarPath) {
            await this.assetRepository.update({ id, sidecarPath });
        }
        return enum_1.JobStatus.Success;
    }
    getSidecarCandidates({ sidecarPath, originalPath }) {
        const candidates = [];
        if (sidecarPath) {
            candidates.push(sidecarPath);
        }
        const assetPath = (0, node_path_1.parse)(originalPath);
        candidates.push(`${originalPath}.xmp`, `${(0, node_path_1.join)(assetPath.dir, assetPath.name)}.xmp`);
        return candidates;
    }
    getImageDimensions(exifTags) {
        let [width, height] = exifTags.ImageSize?.toString()
            ?.split('x')
            ?.map((dim) => Number.parseInt(dim) || undefined) ?? [];
        if (!width || !height) {
            [width, height] = [exifTags.ImageWidth, exifTags.ImageHeight];
        }
        return { width, height };
    }
    getExifTags(asset) {
        if (!asset.sidecarPath && asset.type === enum_1.AssetType.Image) {
            return this.metadataRepository.readTags(asset.originalPath);
        }
        return this.mergeExifTags(asset);
    }
    async mergeExifTags(asset) {
        const [mediaTags, sidecarTags, videoTags] = await Promise.all([
            this.metadataRepository.readTags(asset.originalPath),
            asset.sidecarPath ? this.metadataRepository.readTags(asset.sidecarPath) : null,
            asset.type === enum_1.AssetType.Video ? this.getVideoTags(asset.originalPath) : null,
        ]);
        if (sidecarTags) {
            const result = firstDateTime(sidecarTags);
            const sidecarDate = result?.dateTime;
            if (sidecarDate) {
                for (const tag of EXIF_DATE_TAGS) {
                    delete mediaTags[tag];
                }
            }
        }
        delete mediaTags.Duration;
        delete sidecarTags?.Duration;
        return { ...mediaTags, ...videoTags, ...sidecarTags };
    }
    getTagList(exifTags) {
        let tags;
        if (exifTags.TagsList) {
            tags = exifTags.TagsList.map(String);
        }
        else if (exifTags.HierarchicalSubject) {
            tags = exifTags.HierarchicalSubject.map((tag) => typeof tag === 'number'
                ? String(tag)
                : tag
                    .split('|')
                    .map((tag) => tag.replaceAll('/', '|'))
                    .join('/'));
        }
        else if (exifTags.Keywords) {
            let keywords = exifTags.Keywords;
            if (!Array.isArray(keywords)) {
                keywords = [keywords];
            }
            tags = keywords.map(String);
        }
        else {
            tags = [];
        }
        return tags;
    }
    async applyTagList(asset, exifTags) {
        const tags = this.getTagList(exifTags);
        const results = await (0, tag_1.upsertTags)(this.tagRepository, { userId: asset.ownerId, tags });
        await this.tagRepository.replaceAssetTags(asset.id, results.map((tag) => tag.id));
    }
    isMotionPhoto(asset, tags) {
        return asset.type === enum_1.AssetType.Image && !!(tags.MotionPhoto || tags.MicroVideo);
    }
    async applyMotionPhotos(asset, tags, dates, stats) {
        const isMotionPhoto = tags.MotionPhoto;
        const isMicroVideo = tags.MicroVideo;
        const videoOffset = tags.MicroVideoOffset;
        const hasMotionPhotoVideo = tags.MotionPhotoVideo;
        const hasEmbeddedVideoFile = tags.EmbeddedVideoType === 'MotionPhoto_Data' && tags.EmbeddedVideoFile;
        const directory = Array.isArray(tags.ContainerDirectory)
            ? tags.ContainerDirectory
            : null;
        let length = 0;
        let padding = 0;
        if (isMotionPhoto && directory) {
            for (const entry of directory) {
                if (entry?.Item?.Semantic === 'MotionPhoto') {
                    length = entry.Item.Length ?? 0;
                    padding = entry.Item.Padding ?? 0;
                    break;
                }
            }
        }
        if (isMicroVideo && typeof videoOffset === 'number') {
            length = videoOffset;
        }
        if (!length && !hasEmbeddedVideoFile && !hasMotionPhotoVideo) {
            return;
        }
        this.logger.debug(`Starting motion photo video extraction for asset ${asset.id}: ${asset.originalPath}`);
        try {
            const position = stats.size - length - padding;
            let video;
            if (hasMotionPhotoVideo) {
                video = await this.metadataRepository.extractBinaryTag(asset.originalPath, 'MotionPhotoVideo');
            }
            else if (hasEmbeddedVideoFile) {
                video = await this.metadataRepository.extractBinaryTag(asset.originalPath, 'EmbeddedVideoFile');
            }
            else {
                video = await this.storageRepository.readFile(asset.originalPath, {
                    buffer: Buffer.alloc(length),
                    position,
                    length,
                });
            }
            const checksum = this.cryptoRepository.hashSha1(video);
            const checksumQuery = { ownerId: asset.ownerId, libraryId: asset.libraryId ?? undefined, checksum };
            let motionAsset = await this.assetRepository.getByChecksum(checksumQuery);
            let isNewMotionAsset = false;
            if (!motionAsset) {
                try {
                    const motionAssetId = this.cryptoRepository.randomUUID();
                    motionAsset = await this.assetRepository.create({
                        id: motionAssetId,
                        libraryId: asset.libraryId,
                        type: enum_1.AssetType.Video,
                        fileCreatedAt: dates.dateTimeOriginal,
                        fileModifiedAt: stats.mtime,
                        localDateTime: dates.localDateTime,
                        checksum,
                        ownerId: asset.ownerId,
                        originalPath: storage_core_1.StorageCore.getAndroidMotionPath(asset, motionAssetId),
                        originalFileName: `${(0, node_path_1.parse)(asset.originalFileName).name}.mp4`,
                        visibility: enum_1.AssetVisibility.Hidden,
                        deviceAssetId: 'NONE',
                        deviceId: 'NONE',
                    });
                    isNewMotionAsset = true;
                    if (!asset.isExternal) {
                        await this.userRepository.updateUsage(asset.ownerId, video.byteLength);
                    }
                }
                catch (error) {
                    if (!(0, database_1.isAssetChecksumConstraint)(error)) {
                        throw error;
                    }
                    motionAsset = await this.assetRepository.getByChecksum(checksumQuery);
                    if (!motionAsset) {
                        this.logger.warn(`Unable to find existing motion video asset for ${asset.id}: ${asset.originalPath}`);
                        return;
                    }
                }
            }
            if (!isNewMotionAsset) {
                this.logger.debugFn(() => {
                    const base64Checksum = checksum.toString('base64');
                    return `Motion asset with checksum ${base64Checksum} already exists for asset ${asset.id}: ${asset.originalPath}`;
                });
            }
            if (motionAsset.visibility === enum_1.AssetVisibility.Timeline) {
                await this.assetRepository.update({
                    id: motionAsset.id,
                    visibility: enum_1.AssetVisibility.Hidden,
                });
                this.logger.log(`Hid unlinked motion photo video asset (${motionAsset.id})`);
            }
            if (asset.livePhotoVideoId !== motionAsset.id) {
                await this.assetRepository.update({ id: asset.id, livePhotoVideoId: motionAsset.id });
                if (asset.livePhotoVideoId) {
                    await this.jobRepository.queue({
                        name: enum_1.JobName.AssetDelete,
                        data: { id: asset.livePhotoVideoId, deleteOnDisk: true },
                    });
                    this.logger.log(`Removed old motion photo video asset (${asset.livePhotoVideoId})`);
                }
            }
            const existsOnDisk = await this.storageRepository.checkFileExists(motionAsset.originalPath);
            if (!existsOnDisk) {
                this.storageCore.ensureFolders(motionAsset.originalPath);
                await this.storageRepository.createFile(motionAsset.originalPath, video);
                this.logger.log(`Wrote motion photo video to ${motionAsset.originalPath}`);
                await this.handleMetadataExtraction({ id: motionAsset.id });
                await this.jobRepository.queue({ name: enum_1.JobName.AssetEncodeVideo, data: { id: motionAsset.id } });
            }
            this.logger.debug(`Finished motion photo video extraction for asset ${asset.id}: ${asset.originalPath}`);
        }
        catch (error) {
            this.logger.error(`Failed to extract motion video for ${asset.id}: ${asset.originalPath}: ${error}`, error?.stack);
        }
    }
    hasTaggedFaces(tags) {
        return (tags.RegionInfo !== undefined && tags.RegionInfo.AppliedToDimensions && tags.RegionInfo.RegionList.length > 0);
    }
    orientRegionInfo(regionInfo, orientation) {
        if (orientation === undefined || orientation === enum_1.ExifOrientation.Horizontal) {
            return regionInfo;
        }
        const isSidewards = [
            enum_1.ExifOrientation.MirrorHorizontalRotate270CW,
            enum_1.ExifOrientation.Rotate90CW,
            enum_1.ExifOrientation.MirrorHorizontalRotate90CW,
            enum_1.ExifOrientation.Rotate270CW,
        ].includes(orientation);
        const adjustedAppliedToDimensions = isSidewards
            ? {
                ...regionInfo.AppliedToDimensions,
                W: regionInfo.AppliedToDimensions.H,
                H: regionInfo.AppliedToDimensions.W,
            }
            : regionInfo.AppliedToDimensions;
        const adjustedRegionList = regionInfo.RegionList.map((region) => {
            let { X, Y, W, H } = region.Area;
            switch (orientation) {
                case enum_1.ExifOrientation.MirrorHorizontal: {
                    X = 1 - X;
                    break;
                }
                case enum_1.ExifOrientation.Rotate180: {
                    [X, Y] = [1 - X, 1 - Y];
                    break;
                }
                case enum_1.ExifOrientation.MirrorVertical: {
                    Y = 1 - Y;
                    break;
                }
                case enum_1.ExifOrientation.MirrorHorizontalRotate270CW: {
                    [X, Y] = [Y, X];
                    break;
                }
                case enum_1.ExifOrientation.Rotate90CW: {
                    [X, Y] = [1 - Y, X];
                    break;
                }
                case enum_1.ExifOrientation.MirrorHorizontalRotate90CW: {
                    [X, Y] = [1 - Y, 1 - X];
                    break;
                }
                case enum_1.ExifOrientation.Rotate270CW: {
                    [X, Y] = [Y, 1 - X];
                    break;
                }
            }
            if (isSidewards) {
                [W, H] = [H, W];
            }
            return {
                ...region,
                Area: { ...region.Area, X, Y, W, H },
            };
        });
        return {
            ...regionInfo,
            AppliedToDimensions: adjustedAppliedToDimensions,
            RegionList: adjustedRegionList,
        };
    }
    async applyTaggedFaces(asset, tags) {
        if (!tags.RegionInfo?.AppliedToDimensions || tags.RegionInfo.RegionList.length === 0) {
            return;
        }
        const facesToAdd = [];
        const existingNames = await this.personRepository.getDistinctNames(asset.ownerId, { withHidden: true });
        const existingNameMap = new Map(existingNames.map(({ id, name }) => [name.toLowerCase(), id]));
        const missing = [];
        const missingWithFaceAsset = [];
        const adjustedRegionInfo = this.orientRegionInfo(tags.RegionInfo, tags.Orientation);
        const imageWidth = adjustedRegionInfo.AppliedToDimensions.W;
        const imageHeight = adjustedRegionInfo.AppliedToDimensions.H;
        for (const region of adjustedRegionInfo.RegionList) {
            if (!region.Name) {
                continue;
            }
            const loweredName = region.Name.toLowerCase();
            const personId = existingNameMap.get(loweredName) || this.cryptoRepository.randomUUID();
            const face = {
                id: this.cryptoRepository.randomUUID(),
                personId,
                assetId: asset.id,
                imageWidth,
                imageHeight,
                boundingBoxX1: Math.floor((region.Area.X - region.Area.W / 2) * imageWidth),
                boundingBoxY1: Math.floor((region.Area.Y - region.Area.H / 2) * imageHeight),
                boundingBoxX2: Math.floor((region.Area.X + region.Area.W / 2) * imageWidth),
                boundingBoxY2: Math.floor((region.Area.Y + region.Area.H / 2) * imageHeight),
                sourceType: enum_1.SourceType.Exif,
            };
            facesToAdd.push(face);
            if (!existingNameMap.has(loweredName)) {
                missing.push({ id: personId, ownerId: asset.ownerId, name: region.Name });
                missingWithFaceAsset.push({ id: personId, ownerId: asset.ownerId, faceAssetId: face.id });
            }
        }
        if (missing.length > 0) {
            this.logger.debugFn(() => `Creating missing persons: ${missing.map((p) => `${p.name}/${p.id}`)}`);
            const newPersonIds = await this.personRepository.createAll(missing);
            const jobs = newPersonIds.map((id) => ({ name: enum_1.JobName.PersonGenerateThumbnail, data: { id } }));
            await this.jobRepository.queueAll(jobs);
        }
        const facesToRemove = asset.faces.filter((face) => face.sourceType === enum_1.SourceType.Exif).map((face) => face.id);
        if (facesToRemove.length > 0) {
            this.logger.debug(`Removing ${facesToRemove.length} faces for asset ${asset.id}: ${asset.originalPath}`);
        }
        if (facesToAdd.length > 0) {
            this.logger.debug(`Creating ${facesToAdd.length} faces from metadata for asset ${asset.id}: ${asset.originalPath}`);
        }
        if (facesToRemove.length > 0 || facesToAdd.length > 0) {
            await this.personRepository.refreshFaces(facesToAdd, facesToRemove);
        }
        if (missingWithFaceAsset.length > 0) {
            await this.personRepository.updateAll(missingWithFaceAsset);
        }
    }
    getDates(asset, exifTags, stats) {
        const result = firstDateTime(exifTags);
        const tag = result?.tag;
        const dateTime = result?.dateTime;
        this.logger.verbose(`Date and time is ${dateTime} using exifTag ${tag} for asset ${asset.id}: ${asset.originalPath}`);
        let timeZone = exifTags.tz ?? null;
        if (timeZone == null && dateTime?.rawValue?.endsWith('+00:00')) {
            timeZone = 'UTC+0';
        }
        if (timeZone) {
            this.logger.verbose(`Found timezone ${timeZone} via ${exifTags.tzSource} for asset ${asset.id}: ${asset.originalPath}`);
        }
        else {
            this.logger.debug(`No timezone information found for asset ${asset.id}: ${asset.originalPath}`);
        }
        let dateTimeOriginal = dateTime?.toDateTime();
        if (dateTimeOriginal && !dateTime?.hasZone) {
            dateTimeOriginal = dateTimeOriginal.setZone('UTC', { keepLocalTime: true });
        }
        dateTimeOriginal = dateTimeOriginal?.setZone(timeZone ?? 'UTC');
        let localDateTime = dateTimeOriginal?.setZone('UTC', { keepLocalTime: true });
        if (!localDateTime || !dateTimeOriginal) {
            const earliestDate = luxon_1.DateTime.fromMillis(Math.min(asset.fileCreatedAt.getTime(), stats.birthtimeMs ? Math.min(stats.mtimeMs, stats.birthtimeMs) : stats.mtime.getTime()));
            this.logger.debug(`No exif date time found, falling back on ${earliestDate.toISO()}, earliest of file creation and modification for asset ${asset.id}: ${asset.originalPath}`);
            dateTimeOriginal = localDateTime = earliestDate;
        }
        this.logger.verbose(`Found local date time ${localDateTime.toISO()} for asset ${asset.id}: ${asset.originalPath}`);
        return {
            timeZone,
            localDateTime: localDateTime.toJSDate(),
            dateTimeOriginal: dateTimeOriginal.toJSDate(),
        };
    }
    hasGeo(tags) {
        const lat = Number(tags.GPSLatitude);
        const lng = Number(tags.GPSLongitude);
        return !Number.isNaN(lat) && !Number.isNaN(lng) && (lat !== 0 || lng !== 0);
    }
    getAutoStackId(tags) {
        if (!tags) {
            return null;
        }
        return tags.BurstID ?? tags.BurstUUID ?? tags.CameraBurstID ?? tags.MediaUniqueID ?? null;
    }
    getBitsPerSample(tags) {
        const bitDepthTags = [
            tags.BitsPerSample,
            tags.ComponentBitDepth,
            tags.ImagePixelDepth,
            tags.BitDepth,
            tags.ColorBitDepth,
        ].map((tag) => (typeof tag === 'string' ? Number.parseInt(tag) : tag));
        let bitsPerSample = bitDepthTags.find((tag) => typeof tag === 'number' && !Number.isNaN(tag)) ?? null;
        if (bitsPerSample && bitsPerSample >= 24 && bitsPerSample % 3 === 0) {
            bitsPerSample /= 3;
        }
        return bitsPerSample;
    }
    async getVideoTags(originalPath) {
        const { videoStreams, format } = await this.mediaRepository.probe(originalPath);
        const tags = {};
        if (videoStreams[0]) {
            switch (videoStreams[0].rotation) {
                case -90: {
                    tags.Orientation = enum_1.ExifOrientation.Rotate90CW;
                    break;
                }
                case 0: {
                    tags.Orientation = enum_1.ExifOrientation.Horizontal;
                    break;
                }
                case 90: {
                    tags.Orientation = enum_1.ExifOrientation.Rotate270CW;
                    break;
                }
                case 180: {
                    tags.Orientation = enum_1.ExifOrientation.Rotate180;
                    break;
                }
            }
        }
        if (format.duration) {
            tags.Duration = luxon_1.Duration.fromObject({ seconds: format.duration }).toFormat('hh:mm:ss.SSS');
        }
        return tags;
    }
};
exports.MetadataService = MetadataService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AppBootstrap', workers: [enum_1.ImmichWorker.Microservices] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "onBootstrap", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AppShutdown' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "onShutdown", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigInit', workers: [enum_1.ImmichWorker.Microservices] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], MetadataService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigUpdate', workers: [enum_1.ImmichWorker.Microservices], server: true }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], MetadataService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetExtractMetadataQueueAll, queue: enum_1.QueueName.MetadataExtraction }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleQueueMetadataExtraction", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetExtractMetadata, queue: enum_1.QueueName.MetadataExtraction }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleMetadataExtraction", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.SidecarQueueAll, queue: enum_1.QueueName.Sidecar }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleQueueSidecar", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.SidecarCheck, queue: enum_1.QueueName.Sidecar }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleSidecarCheck", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AssetTag' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleTagAsset", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AssetUntag' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleUntagAsset", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.SidecarWrite, queue: enum_1.QueueName.Sidecar }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleSidecarWrite", null);
exports.MetadataService = MetadataService = __decorate([
    (0, common_1.Injectable)()
], MetadataService);
//# sourceMappingURL=metadata.service.js.map