import { ModuleRef } from '@nestjs/core';
import { ClassConstructor } from 'class-transformer';
import { Socket } from 'socket.io';
import { SystemConfig } from 'src/config';
import { Asset } from 'src/database';
import { AuthDto } from 'src/dtos/auth.dto';
import { JobStatus, QueueName, UserAvatarColor, UserStatus } from 'src/enum';
import { ConfigRepository } from 'src/repositories/config.repository';
import { LoggingRepository } from 'src/repositories/logging.repository';
import { JobItem, JobSource } from 'src/types';
type EventMap = {
    AppBootstrap: [];
    AppShutdown: [];
    AppRestart: [AppRestartEvent];
    ConfigInit: [{
        newConfig: SystemConfig;
    }];
    ConfigUpdate: [
        {
            newConfig: SystemConfig;
            oldConfig: SystemConfig;
        }
    ];
    ConfigValidate: [{
        newConfig: SystemConfig;
        oldConfig: SystemConfig;
    }];
    AlbumUpdate: [{
        id: string;
        recipientId: string;
    }];
    AlbumInvite: [{
        id: string;
        userId: string;
    }];
    AssetCreate: [{
        asset: Asset;
    }];
    AssetTag: [{
        assetId: string;
    }];
    AssetUntag: [{
        assetId: string;
    }];
    AssetHide: [{
        assetId: string;
        userId: string;
    }];
    AssetShow: [{
        assetId: string;
        userId: string;
    }];
    AssetTrash: [{
        assetId: string;
        userId: string;
    }];
    AssetDelete: [{
        assetId: string;
        userId: string;
    }];
    AssetMetadataExtracted: [{
        assetId: string;
        userId: string;
        source?: JobSource;
    }];
    AssetTrashAll: [{
        assetIds: string[];
        userId: string;
    }];
    AssetDeleteAll: [{
        assetIds: string[];
        userId: string;
    }];
    AssetRestoreAll: [{
        assetIds: string[];
        userId: string;
    }];
    JobRun: [QueueName, JobItem];
    JobStart: [QueueName, JobItem];
    JobComplete: [QueueName, JobItem];
    JobSuccess: [JobSuccessEvent];
    JobError: [JobErrorEvent];
    QueueStart: [QueueStartEvent];
    SessionDelete: [{
        sessionId: string;
    }];
    StackCreate: [{
        stackId: string;
        userId: string;
    }];
    StackUpdate: [{
        stackId: string;
        userId: string;
    }];
    StackDelete: [{
        stackId: string;
        userId: string;
    }];
    StackDeleteAll: [{
        stackIds: string[];
        userId: string;
    }];
    UserSignup: [{
        notify: boolean;
        id: string;
        password?: string;
    }];
    UserCreate: [UserEvent];
    UserTrash: [UserEvent];
    UserDelete: [UserEvent];
    UserRestore: [UserEvent];
    AuthChangePassword: [{
        userId: string;
        currentSessionId?: string;
        invalidateSessions?: boolean;
    }];
    WebsocketConnect: [{
        userId: string;
    }];
};
export type AppRestartEvent = {
    isMaintenanceMode: boolean;
};
type JobSuccessEvent = {
    job: JobItem;
    response?: JobStatus;
};
type JobErrorEvent = {
    job: JobItem;
    error: Error | any;
};
type QueueStartEvent = {
    name: QueueName;
};
type UserEvent = {
    name: string;
    id: string;
    createdAt: Date;
    updatedAt: Date;
    deletedAt: Date | null;
    status: UserStatus;
    email: string;
    profileImagePath: string;
    isAdmin: boolean;
    shouldChangePassword: boolean;
    avatarColor: UserAvatarColor | null;
    oauthId: string;
    storageLabel: string | null;
    quotaSizeInBytes: number | null;
    quotaUsageInBytes: number;
    profileChangedAt: Date;
};
export type EmitEvent = keyof EventMap;
export type EmitHandler<T extends EmitEvent> = (...args: ArgsOf<T>) => Promise<void> | void;
export type ArgOf<T extends EmitEvent> = EventMap[T][0];
export type ArgsOf<T extends EmitEvent> = EventMap[T];
export type EventItem<T extends EmitEvent> = {
    event: T;
    handler: EmitHandler<T>;
    server: boolean;
};
export type AuthFn = (client: Socket) => Promise<AuthDto>;
export declare class EventRepository {
    private moduleRef;
    private configRepository;
    private logger;
    private emitHandlers;
    constructor(moduleRef: ModuleRef, configRepository: ConfigRepository, logger: LoggingRepository);
    setup({ services }: {
        services: ClassConstructor<unknown>[];
    }): void;
    private addHandler;
    emit<T extends EmitEvent>(event: T, ...args: ArgsOf<T>): Promise<void>;
    onEvent<T extends EmitEvent>(event: {
        name: T;
        args: ArgsOf<T>;
        server: boolean;
    }): Promise<void>;
}
export {};
