"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseTriggerType = exports.hasMask = exports.readTriggers = void 0;
const readTriggers = async (ctx, db) => {
    const triggers = await db
        .selectFrom('pg_trigger as t')
        .innerJoin('pg_proc as p', 't.tgfoid', 'p.oid')
        .innerJoin('pg_namespace as n', 'p.pronamespace', 'n.oid')
        .innerJoin('pg_class as c', 't.tgrelid', 'c.oid')
        .select((eb) => [
        't.tgname as name',
        't.tgenabled as enabled',
        't.tgtype as type',
        't.tgconstraint as _constraint',
        't.tgdeferrable as is_deferrable',
        't.tginitdeferred as is_initially_deferred',
        't.tgargs as arguments',
        't.tgoldtable as referencing_old_table_as',
        't.tgnewtable as referencing_new_table_as',
        eb.fn('pg_get_expr', ['t.tgqual', 't.tgrelid']).as('when_expression'),
        'p.proname as function_name',
        'c.relname as table_name',
    ])
        .where('t.tgisinternal', '=', false)
        .where('n.nspname', '=', ctx.schemaName)
        .execute();
    for (const trigger of triggers) {
        const table = ctx.getTableByName(trigger.table_name);
        if (!table) {
            continue;
        }
        table.triggers.push({
            name: trigger.name,
            tableName: trigger.table_name,
            functionName: trigger.function_name,
            referencingNewTableAs: trigger.referencing_new_table_as ?? undefined,
            referencingOldTableAs: trigger.referencing_old_table_as ?? undefined,
            when: trigger.when_expression,
            synchronize: true,
            ...(0, exports.parseTriggerType)(trigger.type),
        });
    }
};
exports.readTriggers = readTriggers;
const hasMask = (input, mask) => (input & mask) === mask;
exports.hasMask = hasMask;
const parseTriggerType = (type) => {
    const scope = (0, exports.hasMask)(type, 1 << 0) ? 'row' : 'statement';
    let timing = 'after';
    const timingMasks = [
        { mask: 1 << 1, value: 'before' },
        { mask: 1 << 6, value: 'instead of' },
    ];
    for (const { mask, value } of timingMasks) {
        if ((0, exports.hasMask)(type, mask)) {
            timing = value;
            break;
        }
    }
    const actions = [];
    const actionMasks = [
        { mask: 1 << 2, value: 'insert' },
        { mask: 1 << 3, value: 'delete' },
        { mask: 1 << 4, value: 'update' },
        { mask: 1 << 5, value: 'truncate' },
    ];
    for (const { mask, value } of actionMasks) {
        if ((0, exports.hasMask)(type, mask)) {
            actions.push(value);
            break;
        }
    }
    if (actions.length === 0) {
        throw new Error(`Unable to parse trigger type ${type}`);
    }
    return { actions, timing, scope };
};
exports.parseTriggerType = parseTriggerType;
//# sourceMappingURL=trigger.reader.js.map