import { BatchClusterEmitter, ChildEndReason } from "./BatchClusterEmitter";
import type { BatchClusterOptions } from "./BatchClusterOptions";
import type { BatchClusterStats } from "./BatchClusterStats";
import type { BatchProcessOptions } from "./BatchProcessOptions";
import type { ChildProcessFactory } from "./ChildProcessFactory";
import type { CombinedBatchProcessOptions } from "./CombinedBatchProcessOptions";
import { Deferred } from "./Deferred";
import { Task } from "./Task";
export { BatchClusterOptions } from "./BatchClusterOptions";
export { BatchProcess } from "./BatchProcess";
export { Deferred } from "./Deferred";
export * from "./Logger";
export { SimpleParser } from "./Parser";
export { kill, pidExists } from "./Pids";
export { Rate } from "./Rate";
export { Task } from "./Task";
export type { Args } from "./Args";
export type { BatchClusterEmitter, BatchClusterEvents, ChildEndReason, TypedEventEmitter, } from "./BatchClusterEmitter";
export type { WithObserver } from "./BatchClusterOptions";
export type { BatchClusterStats } from "./BatchClusterStats";
export type { BatchProcessOptions } from "./BatchProcessOptions";
export type { ChildProcessFactory } from "./ChildProcessFactory";
export type { CombinedBatchProcessOptions } from "./CombinedBatchProcessOptions";
export type { HealthCheckStrategy } from "./HealthCheckStrategy";
export type { InternalBatchProcessOptions } from "./InternalBatchProcessOptions";
export type { LoggerFunction } from "./Logger";
export type { Parser } from "./Parser";
export type { HealthCheckable, ProcessHealthMonitor, } from "./ProcessHealthMonitor";
export type { TaskOptions } from "./Task";
export type { WhyNotHealthy, WhyNotReady } from "./WhyNotHealthy";
/**
 * BatchCluster instances manage 0 or more homogeneous child processes, and
 * provide the main interface for enqueuing `Task`s via `enqueueTask`.
 *
 * Given the large number of configuration options, the constructor
 * receives a single options hash. The most important of these are the
 * `ChildProcessFactory`, which specifies the factory that creates
 * ChildProcess instances, and `BatchProcessOptions`, which specifies how
 * child tasks can be verified and shut down.
 */
export declare class BatchCluster {
    #private;
    readonly options: CombinedBatchProcessOptions;
    readonly emitter: BatchClusterEmitter;
    constructor(opts: Partial<BatchClusterOptions> & BatchProcessOptions & ChildProcessFactory);
    /**
     * @see BatchClusterEvents
     */
    readonly on: <E extends keyof import("./BatchClusterEmitter").BatchClusterEvents>(eventName: E, listener: (...args: import("./Args").Args<import("./BatchClusterEmitter").BatchClusterEvents[E]>) => void) => BatchClusterEmitter;
    /**
     * @see BatchClusterEvents
     * @since v9.0.0
     */
    readonly off: <E extends keyof import("./BatchClusterEmitter").BatchClusterEvents>(eventName: E, listener: (...args: import("./Args").Args<import("./BatchClusterEmitter").BatchClusterEvents[E]>) => void) => BatchClusterEmitter;
    get ended(): boolean;
    /**
     * Shut down this instance, and all child processes.
     * @param gracefully should an attempt be made to finish in-flight tasks, or
     * should we force-kill child PIDs.
     */
    end(gracefully?: boolean): Deferred<void>;
    /**
     * Submits `task` for processing by a `BatchProcess` instance
     *
     * @return a Promise that is resolved or rejected once the task has been
     * attempted on an idle BatchProcess
     */
    enqueueTask<T>(task: Task<T>): Promise<T>;
    /**
     * @return true if all previously-enqueued tasks have settled
     */
    get isIdle(): boolean;
    /**
     * @return the number of pending tasks
     */
    get pendingTaskCount(): number;
    /**
     * @returns {number} the mean number of tasks completed by child processes
     */
    get meanTasksPerProc(): number;
    /**
     * @return the total number of child processes created by this instance
     */
    get spawnedProcCount(): number;
    /**
     * @return the current number of spawned child processes. Some (or all) may be idle.
     */
    get procCount(): number;
    /**
     * @return the current number of child processes currently servicing tasks
     */
    get busyProcCount(): number;
    get startingProcCount(): number;
    /**
     * @return the current pending Tasks (mostly for testing)
     */
    get pendingTasks(): readonly Task[];
    /**
     * @return the current running Tasks (mostly for testing)
     */
    get currentTasks(): Task[];
    /**
     * For integration tests:
     */
    get internalErrorCount(): number;
    /**
     * Verify that each BatchProcess PID is actually alive.
     *
     * @return the spawned PIDs that are still in the process table.
     */
    pids(): number[];
    /**
     * For diagnostics. Contents may change.
     */
    stats(): BatchClusterStats;
    /**
     * Get ended process counts (used for tests)
     */
    countEndedChildProcs(why: ChildEndReason): number;
    get childEndCounts(): Record<NonNullable<ChildEndReason>, number>;
    /**
     * Shut down any currently-running child processes. New child processes will
     * be started automatically to handle new tasks.
     */
    closeChildProcesses(gracefully?: boolean): Promise<void>;
    /**
     * Reset the maximum number of active child processes to `maxProcs`. Note that
     * this is handled gracefully: child processes are only reduced as tasks are
     * completed.
     */
    setMaxProcs(maxProcs: number): void;
    /**
     * Run maintenance on currently spawned child processes. This method is
     * normally invoked automatically as tasks are enqueued and processed.
     *
     * Only public for tests.
     */
    vacuumProcs(): Promise<void[]>;
}
