"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeHealthCheckStrategy = exports.TaskTimeoutHealthCheck = exports.AgeHealthCheck = exports.FailureCountHealthCheck = exports.IdleTimeHealthCheck = exports.TaskLimitHealthCheck = exports.StreamHealthCheck = exports.LifecycleHealthCheck = void 0;
/**
 * Checks if process has ended or is ending
 */
class LifecycleHealthCheck {
    assess(process) {
        if (process.ended) {
            return "ended";
        }
        else if (process.ending) {
            return "ending";
        }
        return null;
    }
}
exports.LifecycleHealthCheck = LifecycleHealthCheck;
/**
 * Checks if process stdin is available
 */
class StreamHealthCheck {
    assess(process) {
        if (process.proc.stdin == null || process.proc.stdin.destroyed) {
            return "closed";
        }
        return null;
    }
}
exports.StreamHealthCheck = StreamHealthCheck;
/**
 * Checks if process has exceeded task limits
 */
class TaskLimitHealthCheck {
    assess(process, options) {
        if (options.maxTasksPerProcess > 0 &&
            process.taskCount >= options.maxTasksPerProcess) {
            return "worn";
        }
        return null;
    }
}
exports.TaskLimitHealthCheck = TaskLimitHealthCheck;
/**
 * Checks if process has been idle too long
 */
class IdleTimeHealthCheck {
    assess(process, options) {
        if (options.maxIdleMsPerProcess > 0 &&
            process.idleMs > options.maxIdleMsPerProcess) {
            return "idle";
        }
        return null;
    }
}
exports.IdleTimeHealthCheck = IdleTimeHealthCheck;
/**
 * Checks if process has too many failed tasks
 */
class FailureCountHealthCheck {
    assess(process, options) {
        if (options.maxFailedTasksPerProcess > 0 &&
            process.failedTaskCount >= options.maxFailedTasksPerProcess) {
            return "broken";
        }
        return null;
    }
}
exports.FailureCountHealthCheck = FailureCountHealthCheck;
/**
 * Checks if process is too old
 */
class AgeHealthCheck {
    assess(process, options) {
        if (options.maxProcAgeMillis > 0 &&
            process.start + options.maxProcAgeMillis < Date.now()) {
            return "old";
        }
        return null;
    }
}
exports.AgeHealthCheck = AgeHealthCheck;
/**
 * Checks if current task has timed out
 */
class TaskTimeoutHealthCheck {
    assess(process, options) {
        var _a, _b;
        if (options.taskTimeoutMillis > 0 &&
            ((_b = (_a = process.currentTask) === null || _a === void 0 ? void 0 : _a.runtimeMs) !== null && _b !== void 0 ? _b : 0) > options.taskTimeoutMillis) {
            return "timeout";
        }
        return null;
    }
}
exports.TaskTimeoutHealthCheck = TaskTimeoutHealthCheck;
/**
 * Composite strategy that runs all health checks in order of priority
 */
class CompositeHealthCheckStrategy {
    constructor() {
        this.strategies = [
            new LifecycleHealthCheck(),
            new StreamHealthCheck(),
            new TaskLimitHealthCheck(),
            new IdleTimeHealthCheck(),
            new FailureCountHealthCheck(),
            new AgeHealthCheck(),
            new TaskTimeoutHealthCheck(),
        ];
    }
    assess(process, options) {
        for (const strategy of this.strategies) {
            const result = strategy.assess(process, options);
            if (result != null) {
                return result;
            }
        }
        return null;
    }
}
exports.CompositeHealthCheckStrategy = CompositeHealthCheckStrategy;
//# sourceMappingURL=HealthCheckStrategy.js.map