"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyOptions = verifyOptions;
const BatchClusterOptions_1 = require("./BatchClusterOptions");
const String_1 = require("./String");
/**
 * Verifies and sanitizes the provided options for BatchCluster.
 *
 * It merges partial options with default BatchClusterOptions,
 * converts pass/fail strings to RegExp, and validates various constraints.
 *
 * @param opts - The partial options to verify. These are merged with default
 *   BatchClusterOptions.
 * @returns The fully verified and sanitized options.
 * @throws Error if any options are invalid.
 */
function verifyOptions(opts) {
    const result = {
        ...new BatchClusterOptions_1.BatchClusterOptions(),
        ...opts,
        passRE: toRe(opts.pass),
        failRE: toRe(opts.fail),
    };
    const errors = [];
    function notBlank(fieldName) {
        const v = (0, String_1.toS)(result[fieldName]);
        if ((0, String_1.blank)(v)) {
            errors.push(fieldName + " must not be blank");
        }
    }
    function gte(fieldName, value, why) {
        const v = result[fieldName];
        if (v < value) {
            const msg = `${fieldName} must be greater than or equal to ${value}${(0, String_1.blank)(why) ? "" : ": " + why}`;
            errors.push(msg);
        }
    }
    notBlank("versionCommand");
    notBlank("pass");
    notBlank("fail");
    gte("maxTasksPerProcess", 1);
    gte("maxProcs", 1);
    if (opts.maxProcAgeMillis != null &&
        opts.maxProcAgeMillis > 0 &&
        result.taskTimeoutMillis) {
        gte("maxProcAgeMillis", Math.max(result.spawnTimeoutMillis, result.taskTimeoutMillis), `the max value of spawnTimeoutMillis (${result.spawnTimeoutMillis}) and taskTimeoutMillis (${result.taskTimeoutMillis})`);
    }
    // 0 disables:
    gte("minDelayBetweenSpawnMillis", 0);
    gte("onIdleIntervalMillis", 0);
    gte("endGracefulWaitTimeMillis", 0);
    gte("maxReasonableProcessFailuresPerMinute", 0);
    gte("streamFlushMillis", 0);
    if (errors.length > 0) {
        throw new Error("BatchCluster was given invalid options: " + errors.join("; "));
    }
    return result;
}
function escapeRegExp(s) {
    return (0, String_1.toS)(s).replace(/[-.,\\^$*+?()|[\]{}]/g, "\\$&");
}
function toRe(s) {
    return s instanceof RegExp
        ? s
        : (0, String_1.blank)(s)
            ? /$./
            : s.includes("*")
                ? new RegExp(s
                    .split("*")
                    .map((ea) => escapeRegExp(ea))
                    .join(".*"))
                : new RegExp(escapeRegExp(s));
}
//# sourceMappingURL=OptionsVerifier.js.map