import { Maybe } from "./Maybe";
type Direction = "N" | "S" | "E" | "W";
type CoordinateFormat = "DMS" | "DM" | "D";
interface Coordinate {
    decimal: number;
    degrees: number;
    minutes: number | undefined;
    seconds: number | undefined;
    direction: Direction | undefined;
    format: CoordinateFormat;
    remainder: string | undefined;
}
export interface CoordinateResult {
    latitude: number;
    longitude: number;
}
/**
 * Parses a string containing both latitude and longitude coordinates.
 * @param input - String containing both coordinates
 * @returns Object containing latitude and longitude in decimal degrees, or undefined if parsing fails
 * @throws CoordinateParseError if the input format is invalid
 */
export declare function parseCoordinates(input: string): CoordinateResult;
/**
 * Parses a coordinate string in decimal degrees format.
 * @param input - String containing a single coordinate
 * @returns Object containing degrees and direction, or undefined if parsing fails
 * @throws CoordinateParseError if the format is not decimal degrees or direction is missing
 */
export declare function parseDecimalCoordinate(input: string): {
    decimal: number;
    direction: Direction;
} | undefined;
/**
 * Parses a single coordinate string into its components.
 * @param input - String containing a single coordinate
 * @param expectRemainders - If true, allow additional text after the coordinate
 * @returns Parsed coordinate object
 * @throws CoordinateParseError if the format is invalid
 */
export declare function parseCoordinate(input: string, expectRemainders?: boolean): Coordinate;
export type CoordinateType = "Latitude" | "Longitude";
export interface CoordinateConfig {
    value: number;
    ref: string | undefined;
    geoValue: number | undefined;
    expectedRefPositive: "N" | "E";
    expectedRefNegative: "S" | "W";
    max: 90 | 180;
    coordinateType: CoordinateType;
}
export declare function roundGpsDecimal(decimal: number): number;
export declare function parsePosition(position: Maybe<string>): Maybe<[number, number]>;
export declare function processCoordinate(config: CoordinateConfig, warnings: string[]): {
    value: number;
    ref: string;
    isInvalid: boolean;
};
export {};
