import { DateTime, DateTimeJSOptions, DurationLike, ToISOTimeOptions, Zone, ZoneOptions } from "luxon";
import { Maybe } from "./Maybe";
/**
 * Encapsulates encoding and decoding EXIF date and time strings,
 * along with timezone handling functionality.
 *
 * Key features:
 * - Parses datetime strings in various formats (EXIF strict/loose, ISO)
 * - Supports timezone inference, conversion, and matching between dates
 * - Preserves original string representations when available
 * - Provides conversion to/from multiple datetime formats (Luxon DateTime, JS
 *   Date, ISO strings)
 * - Supports serialization/deserialization via JSON (see
 *   {@link ExifDateTime.fromJSON})
 *
 * EXIF datetime strings don't typically include timezone information. This
 * class provides mechanisms to associate timezone data from other EXIF tags
 * (like GPS position or timezone offset), and distinguishes between explicitly
 * set and inferred timezone information.
 */
export declare class ExifDateTime {
    #private;
    readonly year: number;
    readonly month: number;
    readonly day: number;
    readonly hour: number;
    readonly minute: number;
    readonly second: number;
    readonly millisecond?: number | undefined;
    readonly tzoffsetMinutes?: number | undefined;
    readonly rawValue?: string | undefined;
    readonly zoneName?: string | undefined;
    readonly inferredZone?: boolean | undefined;
    static from(exifOrIso: Maybe<string | ExifDateTime>, defaultZone?: Maybe<string>): Maybe<ExifDateTime>;
    static fromISO(iso: string, defaultZone?: Maybe<string>): Maybe<ExifDateTime>;
    /**
     * Try to parse a date-time string from EXIF. If there is not both a date
     * and a time component, returns `undefined`.
     *
     * @param text from EXIF metadata
     * @param defaultZone a "zone name" to use as a backstop, or default, if
     * `text` doesn't specify a zone. This may be IANA-formatted, like
     * "America/Los_Angeles", or an offset, like "UTC-3". See
     * `offsetMinutesToZoneName`.
     */
    static fromEXIF(text: string, defaultZone?: Maybe<string>): Maybe<ExifDateTime>;
    /**
     * Parse the given date-time string, EXIF-formatted.
     *
     * @param text from EXIF metadata, in `y:M:d H:m:s` format (with optional
     * sub-seconds and/or timezone)
  
     * @param defaultZone a "zone name" to use as a backstop, or default, if
     * `text` doesn't specify a zone. This may be IANA-formatted, like
     * "America/Los_Angeles", or an offset, like "UTC-3". See
     * `offsetMinutesToZoneName`.
     */
    static fromExifStrict(text: unknown, defaultZone?: Maybe<string>): Maybe<ExifDateTime>;
    static fromExifLoose(text: unknown, defaultZone?: Maybe<string>): Maybe<ExifDateTime>;
    static fromDateTime(dt: Maybe<DateTime>, opts?: {
        rawValue?: Maybe<string>;
        unsetMilliseconds?: boolean;
        inferredZone?: Maybe<boolean>;
    }): Maybe<ExifDateTime>;
    /**
     * Create an ExifDateTime from a number of milliseconds since the epoch
     * (meaning since 1 January 1970 00:00:00 UTC). Uses the default zone.
     *
     * @param millis - a number of milliseconds since 1970 UTC
     *
     * @param options.rawValue - the original parsed string input
     * @param options.zone - the zone to place the DateTime into. Defaults to 'local'.
     * @param options.locale - a locale to set on the resulting DateTime instance
     * @param options.outputCalendar - the output calendar to set on the resulting DateTime instance
     * @param options.numberingSystem - the numbering system to set on the resulting DateTime instance
     */
    static fromMillis(millis: number, options?: DateTimeJSOptions & {
        rawValue?: string;
    }): ExifDateTime;
    static now(opts?: DateTimeJSOptions & {
        rawValue?: string;
    }): ExifDateTime;
    readonly zone: Maybe<string>;
    constructor(year: number, month: number, day: number, hour: number, minute: number, second: number, millisecond?: number | undefined, tzoffsetMinutes?: number | undefined, rawValue?: string | undefined, zoneName?: string | undefined, inferredZone?: boolean | undefined);
    get millis(): number | undefined;
    get hasZone(): boolean;
    get unsetMilliseconds(): boolean;
    setZone(zone: string | Zone, opts?: ZoneOptions & {
        inferredZone?: boolean;
    }): Maybe<ExifDateTime>;
    /**
     * CAUTION: This instance will inherit the system timezone if this instance
     * has an unset zone (as Luxon doesn't support "unset" timezones)
     */
    toDateTime(overrideZone?: Maybe<string>): DateTime;
    toEpochSeconds(overrideZone?: Maybe<string>): number;
    toDate(): Date;
    toISOString(options?: ToISOTimeOptions): Maybe<string>;
    toExifString(): string;
    toString(): Maybe<string>;
    /**
     * @return the epoch milliseconds of this
     */
    toMillis(): number;
    get isValid(): boolean;
    toJSON(): {
        _ctor: string;
        year: number;
        month: number;
        day: number;
        hour: number;
        minute: number;
        second: number;
        millisecond: number | undefined;
        tzoffsetMinutes: number | undefined;
        rawValue: string | undefined;
        zoneName: string | undefined;
        inferredZone: boolean | undefined;
    };
    /**
     * @return a new ExifDateTime from the given JSON. Note that this instance **may not be valid**.
     */
    static fromJSON(json: Omit<ReturnType<ExifDateTime["toJSON"]>, "_ctor">): ExifDateTime;
    maybeMatchZone(target: ExifDateTime, maxDeltaMs?: number): Maybe<ExifDateTime>;
    private ifClose;
    plus(duration: DurationLike): Maybe<ExifDateTime>;
}
