"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.globToSqlPattern = exports.useSwagger = exports.routeToErrorMessage = exports.handlePromiseError = exports.isConnectionAborted = exports.isFaceImportEnabled = exports.isDuplicateDetectionEnabled = exports.isFacialRecognitionEnabled = exports.isOcrEnabled = exports.isSmartSearchEnabled = exports.unsetDeep = exports.getKeysDeep = exports.getExternalDomain = exports.getMethodNames = exports.getKeyByValue = exports.isStartUpError = exports.ImmichStartupError = void 0;
exports.getCLIPModelInfo = getCLIPModelInfo;
exports.clamp = clamp;
const swagger_1 = require("@nestjs/swagger");
const lodash_1 = __importDefault(require("lodash"));
const node_fs_1 = require("node:fs");
const node_path_1 = __importDefault(require("node:path"));
const picomatch_1 = __importDefault(require("picomatch"));
const constants_1 = require("../constants");
const sync_dto_1 = require("../dtos/sync.dto");
const enum_1 = require("../enum");
class ImmichStartupError extends Error {
}
exports.ImmichStartupError = ImmichStartupError;
const isStartUpError = (error) => error instanceof ImmichStartupError;
exports.isStartUpError = isStartUpError;
const getKeyByValue = (object, value) => Object.keys(object).find((key) => object[key] === value);
exports.getKeyByValue = getKeyByValue;
const getMethodNames = (instance) => {
    const ctx = Object.getPrototypeOf(instance);
    const methods = [];
    for (const property of Object.getOwnPropertyNames(ctx)) {
        const descriptor = Object.getOwnPropertyDescriptor(ctx, property);
        if (!descriptor || descriptor.get || descriptor.set) {
            continue;
        }
        const handler = instance[property];
        if (typeof handler !== 'function') {
            continue;
        }
        methods.push(property);
    }
    return methods;
};
exports.getMethodNames = getMethodNames;
const getExternalDomain = (server, defaultDomain = 'https://my.immich.app') => server.externalDomain || defaultDomain;
exports.getExternalDomain = getExternalDomain;
const getKeysDeep = (target, path = []) => {
    if (!target || typeof target !== 'object') {
        return [];
    }
    const obj = target;
    const properties = [];
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        if (value === undefined) {
            continue;
        }
        if (lodash_1.default.isObject(value) && !lodash_1.default.isArray(value) && !lodash_1.default.isDate(value)) {
            properties.push(...(0, exports.getKeysDeep)(value, [...path, key]));
            continue;
        }
        properties.push([...path, key].join('.'));
    }
    return properties;
};
exports.getKeysDeep = getKeysDeep;
const unsetDeep = (object, key) => {
    const parts = key.split('.');
    while (parts.length > 0) {
        lodash_1.default.unset(object, parts);
        parts.pop();
        if (!lodash_1.default.isEmpty(lodash_1.default.get(object, parts))) {
            break;
        }
    }
    return lodash_1.default.isEmpty(object) ? undefined : object;
};
exports.unsetDeep = unsetDeep;
const isMachineLearningEnabled = (machineLearning) => machineLearning.enabled;
const isSmartSearchEnabled = (machineLearning) => isMachineLearningEnabled(machineLearning) && machineLearning.clip.enabled;
exports.isSmartSearchEnabled = isSmartSearchEnabled;
const isOcrEnabled = (machineLearning) => isMachineLearningEnabled(machineLearning) && machineLearning.ocr.enabled;
exports.isOcrEnabled = isOcrEnabled;
const isFacialRecognitionEnabled = (machineLearning) => isMachineLearningEnabled(machineLearning) && machineLearning.facialRecognition.enabled;
exports.isFacialRecognitionEnabled = isFacialRecognitionEnabled;
const isDuplicateDetectionEnabled = (machineLearning) => (0, exports.isSmartSearchEnabled)(machineLearning) && machineLearning.duplicateDetection.enabled;
exports.isDuplicateDetectionEnabled = isDuplicateDetectionEnabled;
const isFaceImportEnabled = (metadata) => metadata.faces.import;
exports.isFaceImportEnabled = isFaceImportEnabled;
const isConnectionAborted = (error) => error.code === 'ECONNABORTED';
exports.isConnectionAborted = isConnectionAborted;
const handlePromiseError = (promise, logger) => {
    promise.catch((error) => logger.error(`Promise error: ${error}`, error?.stack));
};
exports.handlePromiseError = handlePromiseError;
function cleanModelName(modelName) {
    const token = modelName.split('/').at(-1);
    if (!token) {
        throw new Error(`Invalid model name: ${modelName}`);
    }
    return token.replaceAll(':', '_');
}
function getCLIPModelInfo(modelName) {
    const modelInfo = constants_1.CLIP_MODEL_INFO[cleanModelName(modelName)];
    if (!modelInfo) {
        throw new Error(`Unknown CLIP model: ${modelName}`);
    }
    return modelInfo;
}
function sortKeys(target) {
    if (!target || typeof target !== 'object' || Array.isArray(target)) {
        return target;
    }
    const result = {};
    const keys = Object.keys(target).toSorted();
    for (const key of keys) {
        result[key] = sortKeys(target[key]);
    }
    return result;
}
const routeToErrorMessage = (methodName) => 'Failed to ' + methodName.replaceAll(/[A-Z]+/g, (letter) => ` ${letter.toLowerCase()}`);
exports.routeToErrorMessage = routeToErrorMessage;
const isSchema = (schema) => {
    if (typeof schema === 'string' || '$ref' in schema) {
        return false;
    }
    return true;
};
const patchOpenAPI = (document) => {
    document.paths = sortKeys(document.paths);
    if (document.components?.schemas) {
        const schemas = document.components.schemas;
        document.components.schemas = sortKeys(schemas);
        for (const [schemaName, schema] of Object.entries(schemas)) {
            if (schema.properties) {
                schema.properties = sortKeys(schema.properties);
                for (const [key, value] of Object.entries(schema.properties)) {
                    if (typeof value === 'string') {
                        continue;
                    }
                    if (isSchema(value) && value.type === 'number' && value.format === 'float') {
                        throw new Error(`Invalid number format: ${schemaName}.${key}=float (use double instead). `);
                    }
                }
                schema.required?.sort();
            }
        }
    }
    for (const [key, value] of Object.entries(document.paths)) {
        const newKey = key.replace('/api/', '/');
        delete document.paths[key];
        document.paths[newKey] = value;
    }
    for (const path of Object.values(document.paths)) {
        const operations = {
            get: path.get,
            put: path.put,
            post: path.post,
            delete: path.delete,
            options: path.options,
            head: path.head,
            patch: path.patch,
            trace: path.trace,
        };
        for (const operation of Object.values(operations)) {
            if (!operation) {
                continue;
            }
            if (operation.summary === '') {
                delete operation.summary;
            }
            if (operation.description === '') {
                delete operation.description;
            }
            if (operation.operationId) {
            }
            if (operation.parameters) {
                operation.parameters = lodash_1.default.orderBy(operation.parameters, 'name');
            }
        }
    }
    return document;
};
const useSwagger = (app, { write }) => {
    const builder = new swagger_1.DocumentBuilder()
        .setTitle('Immich')
        .setDescription('Immich API')
        .setVersion(constants_1.serverVersion.toString())
        .addBearerAuth({
        type: 'http',
        scheme: 'Bearer',
        in: 'header',
    })
        .addCookieAuth(enum_1.ImmichCookie.AccessToken)
        .addApiKey({
        type: 'apiKey',
        in: 'header',
        name: enum_1.ImmichHeader.ApiKey,
    }, enum_1.MetadataKey.ApiKeySecurity)
        .addServer('/api');
    for (const [tag, description] of Object.entries(constants_1.endpointTags)) {
        builder.addTag(tag, description);
    }
    const config = builder.build();
    const options = {
        operationIdFactory: (controllerKey, methodKey) => methodKey,
        extraModels: sync_dto_1.extraSyncModels,
    };
    const specification = swagger_1.SwaggerModule.createDocument(app, config, options);
    const customOptions = {
        swaggerOptions: {
            persistAuthorization: true,
        },
        jsonDocumentUrl: '/api/spec.json',
        yamlDocumentUrl: '/api/spec.yaml',
        customSiteTitle: 'Immich API Documentation',
    };
    swagger_1.SwaggerModule.setup('doc', app, specification, customOptions);
    if (write) {
        const outputPath = node_path_1.default.resolve(process.cwd(), '../open-api/immich-openapi-specs.json');
        (0, node_fs_1.writeFileSync)(outputPath, JSON.stringify(patchOpenAPI(specification), null, 2), { encoding: 'utf8' });
    }
};
exports.useSwagger = useSwagger;
const convertTokenToSqlPattern = (token) => {
    switch (token.type) {
        case 'slash': {
            return '/';
        }
        case 'text': {
            return token.value;
        }
        case 'globstar':
        case 'star': {
            return '%';
        }
        case 'underscore': {
            return String.raw `\_`;
        }
        case 'qmark': {
            return '_';
        }
        case 'dot': {
            return '.';
        }
        default: {
            return '';
        }
    }
};
const globToSqlPattern = (glob) => {
    const tokens = picomatch_1.default.parse(glob).tokens;
    return tokens.map((token) => convertTokenToSqlPattern(token)).join('');
};
exports.globToSqlPattern = globToSqlPattern;
function clamp(value, min, max) {
    return Math.max(min, Math.min(max, value));
}
//# sourceMappingURL=misc.js.map