import * as bc from "batch-cluster";
/**
 * A function that unsubscribes from a Setting change listener.
 * @see Setting.onChange
 */
export type UnsubscribeFunction = () => void;
/**
 * A setting value that can be observed for changes.
 *
 * Each Setting instance manages its own value and listeners, providing
 * a clean, composable alternative to global event buses or proxies.
 */
export declare class Setting<T> {
    #private;
    readonly defaultValue: T;
    constructor(defaultValue: T);
    /** Get the current value */
    get value(): T;
    /** Set a new value, notifying listeners if changed */
    set value(newValue: T);
    /**
     * Subscribe to value changes.
     *
     * @returns Unsubscribe function
     */
    onChange(listener: (oldValue: T, newValue: T) => void): UnsubscribeFunction;
    /** Reset to default value */
    reset(): void;
    /** Allow implicit coercion in conditionals */
    valueOf(): T;
    toString(): string;
}
/**
 * Library-wide configuration settings for exiftool-vendored
 *
 * @see ExifToolOptions for per-instance settings.
 */
export declare const Settings: {
    /**
     * Allow parsing of archaic timezone offsets that are no longer in use.
     *
     * These include historical offsets like:
     * - "-10:30" (Hawaii 1896-1947)
     * - "-04:30" (Venezuela 1912-1965, 2007-2016)
     * - "+04:51" (Bombay until 1955)
     * - and others from
     *   https://en.wikipedia.org/wiki/List_of_tz_database_time_zones
     *
     * **Warning**: Enabling this may lead to incorrect timezone parsing for
     * modern files, as these offsets are not currently used anywhere. Only enable
     * this if you are specifically working with historical photographs or scanned
     * archival material.
     *
     * @default false
     */
    allowArchaicTimezoneOffsets: Setting<boolean>;
    /**
     * Allow parsing of the UTC-12:00 timezone offset ("Baker Island Time") as a
     * valid timezone.
     *
     * This timezone is not used for any populated land, and is disabled by
     * default to prevent incorrect timezone parsing from files with mangled
     * metadata.
     *
     * @default false
     */
    allowBakerIslandTime: Setting<boolean>;
    /**
     * Maximum distance (in minutes) from a valid timezone offset to accept when
     * inferring timezones from GPS or UTC timestamp comparisons.
     *
     * This threshold handles GPS time drift and clock skew. GPS acquisition may
     * lag behind the actual photo time, especially if the GPS fix is old or the
     * camera clock is slightly off.
     *
     * - **15 minutes**: Stricter matching, fewer false positives, but may reject
     *   photos with older GPS fixes
     * - **30 minutes**: More tolerant of GPS lag, recommended for photos that may
     *   have stale GPS data
     *
     * @default 30 minutes
     */
    maxValidOffsetMinutes: Setting<number>;
    /**
     * Logger instance used throughout exiftool-vendored.
     *
     * By default, this is set to ConsoleLogger if NODE_DEBUG=exiftool-vendored is
     * set, otherwise NoLogger.
     *
     * This can be changed at runtime to redirect logging output. When changed,
     * the batch-cluster global logger is also updated for consistency.
     *
     * @default ConsoleLogger or NoLogger based on NODE_DEBUG
     */
    logger: Setting<() => bc.Logger>;
    /** Reset all settings to their default values */
    reset(): void;
};
