"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.timeFormats = timeFormats;
exports.parseDateTime = parseDateTime;
exports.setZone = setZone;
const luxon_1 = require("luxon");
const String_1 = require("./String");
const Timezones_1 = require("./Timezones");
const TimeFmts = [
    // I haven't seen times without padded hours, minutes, or seconds in the
    // wild (yet), so those aren't handled here:
    { fmt: "HH:mm:ss.u", unsetMilliseconds: false },
    { fmt: "HH:mm:ss", unsetMilliseconds: true },
    { fmt: "HH:mm", unsetMilliseconds: true },
];
function* timeFormats(args) {
    const inferredZone = (0, String_1.notBlank)(args.defaultZone);
    for (const prefix of args.formatPrefixes ?? [""]) {
        for (const timeFmt of TimeFmts) {
            yield {
                fmt: prefix + timeFmt.fmt,
                zone: args.defaultZone,
                unsetMilliseconds: timeFmt.unsetMilliseconds,
                inferredZone,
            };
        }
    }
}
function parseDateTime(text, fmts) {
    const s = (0, String_1.toS)(text).trim();
    if (s.length === 0)
        return;
    const extractedZone = (0, Timezones_1.extractZone)(s);
    const input = extractedZone?.leftovers ?? s;
    for (const ea of fmts) {
        const dt = luxon_1.DateTime.fromFormat(input, ea.fmt, {
            setZone: true,
            zone: extractedZone?.tz ?? ea.zone ?? Timezones_1.UnsetZone,
        });
        if (dt?.isValid !== true)
            continue;
        const unsetZone = extractedZone?.tz == null && (dt.zone == null || dt.zone === Timezones_1.UnsetZone);
        let inferredZone = extractedZone?.tz != null || unsetZone ? false : ea.inferredZone;
        if (inferredZone == null) {
            // this is pretty miserable, but luxon doesn't expose _how_ it got
            // the zone, so we have to resort to this hack to see if the zone
            // is inferred:
            const dt2 = luxon_1.DateTime.fromFormat(input, ea.fmt, { setZone: true });
            inferredZone = dt.zone !== dt2.zone;
        }
        return {
            dt,
            fmt: ea.fmt,
            unsetZone,
            inferredZone,
            input,
            unsetMilliseconds: ea.unsetMilliseconds ?? false,
        };
    }
    return;
}
function setZone(args) {
    // This is a bit tricky... We want to keep the local time and just _say_ it
    // was in the zone of the image **but only if we don't already have a zone.**
    // If we _do_ have a zone, assume it was already converted by ExifTool into
    // (probably the system) timezone, which means _don't_ `keepLocalTime`.
    return args.src.setZone(args.zone, {
        keepLocalTime: !args.srcHasZone,
        ...args.opts,
    });
}
//# sourceMappingURL=TimeParsing.js.map